"""Business logic for tasks including recurrence handling."""

from __future__ import annotations

from datetime import date, timedelta
from typing import List, Optional

from ca_task_manager.repositories.task_repo import TaskRepository
from ca_task_manager.models.task import Task, Recurrence


class TaskService:
    def __init__(self, repo: Optional[TaskRepository] = None):
        self.repo = repo or TaskRepository()

    def list_tasks(self) -> List[dict]:
        return self.repo.all()

    def add_task(self, data: dict) -> int:
        # Convert date strings to date objects if necessary
        if isinstance(data.get("due_date"), str):
            data["due_date"] = date.fromisoformat(data["due_date"])
        task = Task(**data)
        # Convert due_date to iso string before storing
        record = task.dict()
        record["due_date"] = task.due_date.isoformat()
        return self.repo.create_task(Task(**record))

    def update_task(self, task_id: int, data: dict) -> bool:
        # If due_date provided as date, convert to iso string
        if "due_date" in data:
            val = data["due_date"]
            if isinstance(val, date):
                data["due_date"] = val.isoformat()
        return self.repo.update_task(task_id, data)

    def get_task(self, task_id: int) -> Optional[dict]:
        return self.repo.get(task_id)

    def handle_recurrence_on_completion(self, task_record: dict) -> Optional[int]:
        """Given a completed recurring task, create the next instance.

        Returns the new task id if created, else None.
        """
        recurrence = task_record.get("recurrence", Recurrence.NONE)
        if isinstance(recurrence, str):
            rec = Recurrence(recurrence)
        else:
            rec = recurrence
        if rec == Recurrence.NONE:
            return None
        # Compute next due date based on recurrence
        due_val = task_record.get("due_date")
        if isinstance(due_val, str):
            current_due = date.fromisoformat(due_val)
        elif isinstance(due_val, date):
            current_due = due_val
        else:
            return None
        if rec == Recurrence.WEEKLY:
            next_due = current_due + timedelta(weeks=1)
        elif rec == Recurrence.MONTHLY:
            # naive approach: add 1 month by adding 30 days; adjust for months using day keep; better to use dateutil, but we avoid extra deps
            next_due = current_due.replace(day=1) + timedelta(days=32)
            next_due = next_due.replace(day=current_due.day)
        elif rec == Recurrence.QUARTERLY:
            next_due = current_due.replace(day=1) + timedelta(days=92)
            next_due = next_due.replace(day=current_due.day)
        elif rec == Recurrence.SEMIANNUAL:
            next_due = current_due.replace(day=1) + timedelta(days=183)
            next_due = next_due.replace(day=current_due.day)
        elif rec == Recurrence.ANNUAL:
            try:
                next_due = current_due.replace(year=current_due.year + 1)
            except ValueError:
                # handle February 29
                next_due = current_due + timedelta(days=365)
        else:
            return None
        # Create new task record copying all fields except id and due_date
        new_data = task_record.copy()
        new_data.pop("id", None)
        new_data["due_date"] = next_due.isoformat()
        # Use same code? For recurring tasks we might keep same task_code; we'll append suffix or rely on service to update tasks. We'll leave as same code
        # Insert new record
        new_task = Task(**new_data)
        # Save iso date again
        rec_dict = new_task.dict()
        rec_dict["due_date"] = new_task.due_date.isoformat()
        return self.repo.create_task(Task(**rec_dict))